import 'package:get_it/get_it.dart';
import 'package:injectable/injectable.dart';
import 'package:shared_preferences/shared_preferences.dart';
import 'package:flutter_secure_storage/flutter_secure_storage.dart';
import 'package:dio/dio.dart';
import 'package:http/http.dart' as http;
import 'package:google_sign_in/google_sign_in.dart';

// Core
import '../core/network/dio_client.dart';
import '../core/network/network_info.dart';
import '../core/services/notification_service.dart';
import '../core/services/global_notification_service.dart';
import '../core/services/stripe_service.dart';
import '../core/services/chart_service.dart';
import '../core/services/futures_websocket_service.dart';
import '../core/services/price_animation_service.dart';
import '../core/services/favorites_service.dart';
import '../core/services/screenshot_service.dart';

// Auth Feature
import '../features/auth/data/datasources/auth_local_data_source.dart';
import '../features/auth/data/datasources/auth_remote_data_source.dart';
import '../features/auth/data/repositories/auth_repository_impl.dart';
import '../features/auth/domain/repositories/auth_repository.dart';
import '../features/auth/domain/usecases/login_usecase.dart';
import '../features/auth/domain/usecases/register_usecase.dart';
import '../features/auth/domain/usecases/logout_usecase.dart';
import '../features/auth/domain/usecases/get_cached_user_usecase.dart';
import '../features/auth/domain/usecases/check_auth_status_usecase.dart';
import '../features/auth/domain/usecases/forgot_password_usecase.dart';
import '../features/auth/domain/usecases/login_with_google_usecase.dart';
import '../features/auth/presentation/bloc/auth_bloc.dart';

// News Feature
import '../features/news/data/datasources/news_remote_datasource.dart';
import '../features/news/data/datasources/news_local_datasource.dart';
import '../features/news/data/repositories/news_repository_impl.dart';
import '../features/news/domain/repositories/news_repository.dart';
import '../features/news/domain/usecases/get_latest_news_usecase.dart';
import '../features/news/domain/usecases/get_trending_news_usecase.dart';
import '../features/news/domain/usecases/search_news_usecase.dart';
import '../features/news/domain/usecases/get_news_categories_usecase.dart';
import '../features/news/presentation/bloc/news_bloc.dart';

// Wallet Feature
import '../features/wallet/data/datasources/wallet_cache_datasource.dart';
import '../features/wallet/data/datasources/wallet_remote_datasource.dart';
import '../features/wallet/data/datasources/currency_price_remote_datasource.dart';
import '../features/wallet/data/repositories/wallet_repository_impl.dart';
import '../features/wallet/data/repositories/currency_price_repository_impl.dart';
import '../features/wallet/domain/repositories/wallet_repository.dart';
import '../features/wallet/domain/repositories/currency_price_repository.dart';
import '../features/wallet/domain/usecases/get_wallets_usecase.dart';
import '../features/wallet/domain/usecases/get_wallets_by_type_usecase.dart';
import '../features/wallet/domain/usecases/get_wallet_usecase.dart';
import '../features/wallet/domain/usecases/get_wallet_by_id_usecase.dart';
import '../features/wallet/domain/usecases/get_wallet_performance_usecase.dart';
import '../features/wallet/domain/usecases/get_currency_price_usecase.dart';
import '../features/wallet/domain/usecases/get_wallet_balance_usecase.dart'
    as wallet;
import '../features/wallet/presentation/bloc/wallet_bloc.dart';
import '../features/wallet/presentation/bloc/currency_price_bloc.dart';

// Deposit Feature
import '../features/wallet/data/datasources/deposit_remote_datasource.dart';
import '../features/wallet/data/repositories/deposit_repository_impl.dart';
import '../features/wallet/domain/repositories/deposit_repository.dart';
import '../features/wallet/domain/usecases/get_currency_options_usecase.dart';
import '../features/wallet/domain/usecases/get_deposit_methods_usecase.dart';
import '../features/wallet/domain/usecases/create_fiat_deposit_usecase.dart';
import '../features/wallet/domain/usecases/create_stripe_payment_intent_usecase.dart';
import '../features/wallet/domain/usecases/verify_stripe_payment_usecase.dart';
import '../features/wallet/domain/usecases/create_paypal_order_usecase.dart';
import '../features/wallet/domain/usecases/verify_paypal_payment_usecase.dart';
import '../features/wallet/presentation/bloc/deposit_bloc.dart';

// SPOT Deposit Feature
import '../features/wallet/data/datasources/spot_deposit_remote_datasource.dart';
import '../features/wallet/data/repositories/spot_deposit_repository_impl.dart';
import '../features/wallet/domain/repositories/spot_deposit_repository.dart';
import '../features/wallet/domain/usecases/get_spot_currencies_usecase.dart';
import '../features/wallet/domain/usecases/get_spot_networks_usecase.dart';
import '../features/wallet/domain/usecases/generate_spot_deposit_address_usecase.dart';
import '../features/wallet/domain/usecases/create_spot_deposit_usecase.dart';
import '../features/wallet/domain/usecases/verify_spot_deposit_usecase.dart';
import '../features/wallet/presentation/bloc/spot_deposit_bloc.dart';

// ECO Deposit Feature
import '../features/wallet/data/datasources/eco_deposit_remote_datasource.dart';
import '../features/wallet/data/repositories/eco_deposit_repository_impl.dart';
import '../features/wallet/domain/repositories/eco_deposit_repository.dart';
import '../features/wallet/domain/usecases/get_eco_currencies_usecase.dart';
import '../features/wallet/domain/usecases/get_eco_tokens_usecase.dart';
import '../features/wallet/domain/usecases/generate_eco_address_usecase.dart';
import '../features/wallet/presentation/bloc/eco_deposit_bloc.dart';

// FUTURES Deposit Feature
import '../features/wallet/data/datasources/futures_deposit_remote_datasource.dart';
import '../features/wallet/data/repositories/futures_deposit_repository_impl.dart';
import '../features/wallet/domain/repositories/futures_deposit_repository.dart';
import '../features/wallet/domain/usecases/get_futures_currencies_usecase.dart';
import '../features/wallet/domain/usecases/get_futures_tokens_usecase.dart';
import '../features/wallet/domain/usecases/generate_futures_address_usecase.dart';
import '../features/wallet/presentation/bloc/futures_deposit_bloc.dart';

// Futures Feature
import '../features/futures/data/datasources/futures_market_remote_datasource.dart';
import '../features/futures/data/datasources/futures_order_remote_datasource.dart';
import '../features/futures/data/datasources/futures_position_remote_datasource.dart';
import '../features/futures/data/repositories/futures_order_repository_impl.dart';
import '../features/futures/data/repositories/futures_position_repository_impl.dart';
import '../features/futures/domain/repositories/futures_order_repository.dart';
import '../features/futures/domain/repositories/futures_position_repository.dart';
import '../features/futures/domain/usecases/get_futures_markets_usecase.dart';
import '../features/futures/domain/usecases/place_futures_order_usecase.dart';
import '../features/futures/domain/usecases/get_futures_positions_usecase.dart';
import '../features/futures/domain/usecases/change_leverage_usecase.dart';
import '../features/futures/presentation/bloc/futures_header_bloc.dart';
import '../features/futures/presentation/bloc/futures_form_bloc.dart';
import '../features/futures/presentation/bloc/futures_positions_bloc.dart';
import '../features/futures/presentation/bloc/futures_orderbook_bloc.dart';

// Profile Feature
import '../features/profile/data/datasources/profile_cache_manager.dart';
import '../features/profile/data/datasources/profile_remote_data_source.dart';
import '../features/profile/data/repositories/profile_repository_impl.dart';
import '../features/profile/domain/repositories/profile_repository.dart';
import '../features/profile/domain/usecases/get_profile_usecase.dart';
import '../features/profile/domain/usecases/update_profile_usecase.dart';
import '../features/profile/domain/usecases/toggle_two_factor_usecase.dart';
import '../features/profile/domain/usecases/generate_two_factor_secret_usecase.dart';
import '../features/profile/domain/usecases/verify_two_factor_setup_usecase.dart';
import '../features/profile/domain/usecases/save_two_factor_setup_usecase.dart';
import '../features/profile/presentation/bloc/profile_bloc.dart';
import '../features/profile/presentation/bloc/two_factor_setup_bloc.dart';
import '../features/profile/data/services/profile_service.dart';

// Transfer feature imports
import '../features/transfer/data/datasources/transfer_remote_datasource.dart';
import '../features/transfer/data/datasources/transfer_remote_datasource_impl.dart';
import '../features/transfer/data/repositories/transfer_repository_impl.dart';
import '../features/transfer/domain/repositories/transfer_repository.dart';
import '../features/transfer/domain/usecases/get_transfer_options_usecase.dart';
import '../features/transfer/domain/usecases/get_transfer_currencies_usecase.dart';
import '../features/transfer/domain/usecases/get_wallet_balance_usecase.dart';
import '../features/transfer/domain/usecases/create_transfer_usecase.dart';
import '../features/transfer/presentation/bloc/transfer_bloc.dart';

// Chart feature imports
import '../features/chart/data/datasources/chart_remote_datasource.dart';
import '../features/chart/data/repositories/chart_repository_impl.dart';
import '../features/chart/domain/repositories/chart_repository.dart';
import '../features/chart/domain/usecases/get_chart_history_usecase.dart';
import '../features/chart/domain/usecases/get_chart_with_volume_usecase.dart';
import '../features/chart/domain/usecases/get_realtime_ticker_usecase.dart';
import '../features/chart/presentation/bloc/chart_bloc.dart';

// Trade Feature
import '../features/trade/presentation/bloc/trading_chart_bloc.dart';
import '../features/trade/presentation/bloc/trading_header_bloc.dart';
import '../features/trade/presentation/bloc/order_book_bloc.dart';
import '../features/trade/presentation/bloc/trading_form_bloc.dart';
import '../features/trade/domain/usecases/get_trading_chart_history_usecase.dart';

// AI Investment Feature
import '../features/trade/presentation/bloc/ai_investment_bloc.dart';
import '../features/trade/domain/usecases/get_ai_investment_plans_usecase.dart';
import '../features/trade/domain/usecases/get_user_ai_investments_usecase.dart';
import '../features/trade/domain/usecases/create_ai_investment_usecase.dart';

// Addons Feature
import '../features/addons/staking/presentation/bloc/staking_bloc.dart';
import '../features/addons/staking/presentation/bloc/position_bloc.dart';

// Blog Feature
import '../features/addons/blog/data/datasources/blog_remote_datasource.dart';
import '../features/addons/blog/data/services/blog_author_service.dart';
import '../features/addons/blog/presentation/bloc/blog_bloc.dart';
import '../features/addons/blog/presentation/bloc/authors_bloc.dart';

// Ecommerce Feature
import '../features/addons/ecommerce/data/datasources/ecommerce_local_datasource.dart';
import '../features/addons/ecommerce/data/datasources/ecommerce_remote_datasource.dart';
import '../features/addons/ecommerce/data/repositories/ecommerce_repository_impl.dart';
import '../features/addons/ecommerce/domain/repositories/ecommerce_repository.dart';
import '../features/addons/ecommerce/domain/usecases/get_products_usecase.dart';
import '../features/addons/ecommerce/domain/usecases/get_categories_usecase.dart';
import '../features/addons/ecommerce/domain/usecases/get_cart_usecase.dart';
import '../features/addons/ecommerce/domain/usecases/add_to_cart_usecase.dart';
import '../features/addons/ecommerce/domain/usecases/update_cart_item_quantity_usecase.dart';
import '../features/addons/ecommerce/domain/usecases/remove_from_cart_usecase.dart';
import '../features/addons/ecommerce/domain/usecases/clear_cart_usecase.dart';
import '../features/addons/ecommerce/domain/usecases/place_order_usecase.dart';
import '../features/addons/ecommerce/presentation/bloc/products/products_bloc.dart';
import '../features/addons/ecommerce/presentation/bloc/cart/cart_bloc.dart';
import '../features/addons/ecommerce/presentation/bloc/shop/shop_bloc.dart';

// P2P Feature
import '../features/addons/p2p/data/datasources/p2p_remote_datasource.dart';
import '../features/addons/p2p/data/datasources/p2p_local_datasource.dart';
import '../features/addons/p2p/data/datasources/p2p_market_remote_datasource.dart';
import '../features/addons/p2p/data/repositories/p2p_offers_repository_impl.dart';
import '../features/addons/p2p/data/repositories/p2p_trades_repository_impl.dart';
import '../features/addons/p2p/data/repositories/p2p_dashboard_repository_impl.dart';
import '../features/addons/p2p/data/repositories/p2p_market_repository_impl.dart';
import '../features/addons/p2p/data/repositories/p2p_payment_methods_repository_impl.dart';
import '../features/addons/p2p/data/repositories/p2p_reviews_repository_impl.dart';
import '../features/addons/p2p/data/repositories/p2p_matching_repository_impl.dart';
import '../features/addons/p2p/domain/repositories/p2p_offers_repository.dart';
import '../features/addons/p2p/domain/repositories/p2p_trades_repository.dart';
import '../features/addons/p2p/domain/repositories/p2p_dashboard_repository.dart';
import '../features/addons/p2p/domain/repositories/p2p_market_repository.dart';
import '../features/addons/p2p/domain/repositories/p2p_payment_methods_repository.dart';
import '../features/addons/p2p/domain/repositories/p2p_reviews_repository.dart';
import '../features/addons/p2p/domain/repositories/p2p_matching_repository.dart';
import '../features/addons/p2p/presentation/bloc/offers/offers_bloc.dart';
import '../features/addons/p2p/presentation/bloc/offers/create_offer_bloc.dart';
import '../features/addons/p2p/presentation/bloc/offers/offer_detail_bloc.dart';
import '../features/addons/p2p/presentation/bloc/trades/trades_bloc.dart';
import '../features/addons/p2p/presentation/bloc/trades/trade_detail_bloc.dart';
import '../features/addons/p2p/presentation/bloc/trades/trade_chat_bloc.dart';
import '../features/addons/p2p/presentation/bloc/market/market_bloc.dart';

// Support Feature
import '../features/support/data/datasources/support_remote_datasource.dart';
import '../features/support/data/datasources/support_websocket_datasource.dart';
import '../features/support/data/repositories/support_repository_impl.dart';
import '../features/support/domain/repositories/support_repository.dart';
import '../features/support/domain/usecases/get_support_tickets_usecase.dart';
import '../features/support/domain/usecases/create_support_ticket_usecase.dart';
import '../features/support/domain/usecases/reply_to_ticket_usecase.dart';
import '../features/support/domain/usecases/live_chat_usecase.dart';
import '../features/support/presentation/bloc/support_tickets_bloc.dart';
import '../features/support/presentation/bloc/live_chat_bloc.dart';
import '../features/support/presentation/bloc/ticket_detail_bloc.dart';

// Creator Feature
import '../features/addons/ico_creator/data/datasources/creator_remote_datasource.dart';

// MLM/Affiliate Feature
import '../features/addons/mlm/data/datasources/mlm_remote_datasource.dart';
import '../features/addons/mlm/data/repositories/mlm_repository_impl.dart';
import '../features/addons/mlm/domain/repositories/mlm_repository.dart';
import '../features/addons/mlm/domain/usecases/get_mlm_dashboard_usecase.dart';
import '../features/addons/mlm/domain/usecases/get_mlm_referrals_usecase.dart';
import '../features/addons/mlm/domain/usecases/get_mlm_rewards_usecase.dart';
import '../features/addons/mlm/domain/usecases/claim_mlm_reward_usecase.dart';
import '../features/addons/mlm/domain/usecases/get_mlm_network_usecase.dart';
import '../features/addons/mlm/domain/usecases/get_mlm_conditions_usecase.dart';
import '../features/addons/mlm/presentation/bloc/mlm_dashboard_bloc.dart';
import '../features/addons/mlm/presentation/bloc/mlm_referrals_bloc.dart';
import '../features/addons/mlm/presentation/bloc/mlm_rewards_bloc.dart';
import '../features/addons/mlm/presentation/bloc/mlm_network_bloc.dart';
import '../features/addons/ico_creator/data/repositories/creator_repository_impl.dart';
import '../features/addons/ico_creator/domain/repositories/creator_repository.dart';
import '../features/addons/ico_creator/presentation/bloc/creator_bloc.dart';
import '../features/addons/ico_creator/domain/usecases/launch_token_usecase.dart';
import '../features/addons/ico_creator/domain/usecases/get_launch_plans_usecase.dart';
import '../features/addons/ico_creator/presentation/bloc/launch_plan_cubit.dart';
import '../features/addons/ico_creator/domain/usecases/get_investors_usecase.dart';
import '../features/addons/ico_creator/domain/usecases/get_creator_stats_usecase.dart';
import '../features/addons/ico_creator/presentation/bloc/investors_cubit.dart';
import '../features/addons/ico_creator/presentation/bloc/stats_cubit.dart';
import '../features/addons/ico_creator/domain/usecases/get_creator_performance_usecase.dart';
import '../features/addons/ico_creator/presentation/bloc/performance_cubit.dart';

// Notification Feature
import '../features/notification/data/datasources/notification_websocket_data_source.dart';

// KYC Feature
import '../features/kyc/data/datasources/kyc_remote_datasource.dart';
import '../features/kyc/data/repositories/kyc_repository_impl.dart';
import '../features/kyc/domain/repositories/kyc_repository.dart';
import '../features/kyc/domain/usecases/get_kyc_levels_usecase.dart';
import '../features/kyc/domain/usecases/get_kyc_applications_usecase.dart';
import '../features/kyc/domain/usecases/submit_kyc_application_usecase.dart';
import '../features/kyc/presentation/bloc/kyc_bloc.dart';

// Theme Feature
import '../features/theme/data/datasources/theme_local_datasource.dart';
import '../features/theme/data/repositories/theme_repository_impl.dart';
import '../features/theme/domain/repositories/theme_repository.dart';
import '../features/theme/domain/usecases/get_saved_theme_usecase.dart';
import '../features/theme/domain/usecases/get_system_theme_usecase.dart';
import '../features/theme/domain/usecases/save_theme_usecase.dart';
import '../features/theme/presentation/bloc/theme_bloc.dart';

// Generated config
import 'injection.config.dart';

final getIt = GetIt.instance;

@InjectableInit()
Future<void> configureDependencies() async {
  // If this function is called more than once (e.g. during Hot Restart) we need
  // to clear any previous registrations to avoid duplicate-registration errors.
  if (getIt.isRegistered<SharedPreferences>()) {
    // Preserve earlier singletons like SharedPreferences by disposing = false
    await getIt.reset(dispose: false);
  }
  // Initialize SharedPreferences first
  final sharedPreferences = await SharedPreferences.getInstance();

  // Register external dependencies that can't be auto-generated
  getIt.registerSingleton<SharedPreferences>(sharedPreferences);
  getIt.registerSingleton<FlutterSecureStorage>(const FlutterSecureStorage());
  getIt.registerSingleton<Dio>(Dio());
  getIt.registerSingleton<http.Client>(http.Client());

  // Manual feature registrations for complex dependencies BEFORE auto-init
  _registerProfileFeature();

  // Initialize the auto-generated dependencies
  getIt.init();

  // Initialize ProfileService after all dependencies are registered
  _initializeProfileService();

  // Manual registrations that need to override auto-generated ones
  _registerAuthFeature();
  _registerWalletFeature();
  _registerDepositFeature();
  _registerSpotDepositFeature();
  _registerEcoDepositFeature();
  _registerFuturesDepositFeature();
  _registerFuturesFeature();
  _registerNotificationFeature();
  _registerTransferFeature();
  _registerChartFeature();
  _registerTradeFeature();
  _registerStakingFeature();
  _registerBlogFeature();
  _registerEcommerceFeature();
  _registerP2PFeature();
  _registerSupportFeature();
  _registerIcoFeature();
  _registerCreatorFeature();
  _registerKycFeature();
  _registerThemeFeature();
  _registerNewsFeature();
  _registerMlmFeature();
}

void _initializeProfileService() {
  // Initialize ProfileService with BlogAuthorService after auto-injection is complete
  try {
    final profileBloc = getIt<ProfileBloc>();
    final blogAuthorService = getIt<BlogAuthorService>();
    getIt<ProfileService>().initialize(profileBloc, blogAuthorService);
    print('🟢 INJECTION: ProfileService initialized successfully');
  } catch (e) {
    print('🔴 INJECTION: Failed to initialize ProfileService: $e');
  }
}

void _registerAuthFeature() {
  // Data sources
  getIt.registerLazySingleton<AuthLocalDataSource>(
    () => AuthLocalDataSourceImpl(
      sharedPreferences: getIt(),
      secureStorage: getIt(),
    ),
  );

  getIt.registerLazySingleton<AuthRemoteDataSource>(
    () => AuthRemoteDataSourceImpl(client: getIt()),
  );

  // Repository
  getIt.registerLazySingleton<AuthRepository>(
    () => AuthRepositoryImpl(
      remoteDataSource: getIt(),
      localDataSource: getIt(),
    ),
  );

  // Use cases
  getIt.registerLazySingleton(() => LoginUseCase(getIt()));
  getIt.registerLazySingleton(() => RegisterUseCase(getIt()));
  getIt.registerLazySingleton(() => LogoutUseCase(getIt()));
  getIt.registerLazySingleton(() => GetCachedUserUseCase(getIt()));
  getIt.registerLazySingleton(() => CheckAuthStatusUseCase(getIt()));
  getIt.registerLazySingleton(() => ForgotPasswordUseCase(getIt()));
  getIt.registerLazySingleton(() => LoginWithGoogleUseCase(getIt()));

  // Bloc (avoid duplicate if generated) - Use lazySingleton to maintain state
  if (!getIt.isRegistered<AuthBloc>()) {
    getIt.registerLazySingleton(
      () => AuthBloc(
        loginUseCase: getIt(),
        loginWithGoogleUseCase: getIt(),
        registerUseCase: getIt(),
        logoutUseCase: getIt(),
        getCachedUserUseCase: getIt(),
        checkAuthStatusUseCase: getIt(),
        forgotPasswordUseCase: getIt(),
        profileService: getIt(),
      ),
    );
  }
}

void _registerWalletFeature() {
  // WalletCacheDataSource and WalletRemoteDataSource are auto-registered
  // Need to manually register wallet repository and its dependencies
  if (!getIt.isRegistered<WalletRepository>()) {
    getIt.registerLazySingleton<WalletRepository>(
      () => WalletRepositoryImpl(
        remoteDataSource: getIt(),
        cacheDataSource: getIt(),
        networkInfo: getIt(),
      ),
    );
  }

  // Currency Price Feature - All classes are auto-registered with @injectable
  // No manual registration needed for:
  // - CurrencyPriceRemoteDataSource (auto-registered)
  // - CurrencyPriceRepository (auto-registered)
  // - GetCurrencyPriceUseCase (auto-registered)
  // - GetCurrencyWalletBalanceUseCase (auto-registered)

  // Prevent duplicate Bloc registration after hot restart
  if (getIt.isRegistered<WalletBloc>()) {
    // If the Bloc was previously registered, remove it to allow re-registration for fresh state
    getIt.unregister<WalletBloc>();
  }

  // CurrencyPriceBloc is auto-registered - don't unregister it

  // Use Cases
  getIt.registerLazySingleton<GetWalletsUseCase>(
    () => GetWalletsUseCase(getIt()),
  );

  getIt.registerLazySingleton<GetWalletsByTypeUseCase>(
    () => GetWalletsByTypeUseCase(getIt()),
  );

  getIt.registerLazySingleton<GetWalletUseCase>(
    () => GetWalletUseCase(getIt()),
  );

  getIt.registerLazySingleton<GetWalletByIdUseCase>(
    () => GetWalletByIdUseCase(getIt()),
  );

  getIt.registerLazySingleton<GetWalletPerformanceUseCase>(
    () => GetWalletPerformanceUseCase(getIt()),
  );

  // Blocs
  getIt.registerFactory<WalletBloc>(
    () => WalletBloc(
      getWalletsUseCase: getIt(),
      getWalletsByTypeUseCase: getIt(),
      getWalletUseCase: getIt(),
      getWalletByIdUseCase: getIt(),
      getWalletPerformanceUseCase: getIt(),
    ),
  );

  // CurrencyPriceBloc is auto-registered with @injectable
  // No manual registration needed
}

void _registerProfileFeature() {
  // Data sources
  getIt.registerLazySingleton<ProfileCacheManager>(
    () => ProfileCacheManager(getIt<FlutterSecureStorage>()),
  );

  getIt.registerLazySingleton<ProfileRemoteDataSource>(
    () => ProfileRemoteDataSourceImpl(dioClient: getIt()),
  );

  // Repository
  getIt.registerLazySingleton<ProfileRepository>(
    () => ProfileRepositoryImpl(
      remoteDataSource: getIt(),
    ),
  );

  // Use cases - Only register non-auto-registered ones
  getIt.registerLazySingleton<GetProfileUseCase>(
    () => GetProfileUseCase(getIt()),
  );

  getIt.registerLazySingleton<UpdateProfileUseCase>(
    () => UpdateProfileUseCase(getIt()),
  );

  getIt.registerLazySingleton<ToggleTwoFactorUseCase>(
    () => ToggleTwoFactorUseCase(getIt()),
  );

  // Two-Factor Setup Use Cases are auto-registered with @injectable
  // No manual registration needed for:
  // - GenerateTwoFactorSecretUseCase
  // - VerifyTwoFactorSetupUseCase
  // - SaveTwoFactorSetupUseCase
  // - TwoFactorSetupBloc

  // Service
  getIt.registerLazySingleton<ProfileService>(
    () => ProfileService.instance,
  );

  // Blocs - Only register non-auto-registered ones
  getIt.registerLazySingleton(
    () => ProfileBloc(
      getProfileUseCase: getIt(),
      updateProfileUseCase: getIt(),
      toggleTwoFactorUseCase: getIt(),
      cacheManager: getIt<ProfileCacheManager>(),
      profileService: getIt<ProfileService>(),
    ),
  );

  // TwoFactorSetupBloc is auto-registered with @injectable
  // No manual registration needed
}

void _registerNotificationFeature() {
  // Manual override for GlobalNotificationService to ensure proper dependency order
  if (getIt.isRegistered<GlobalNotificationService>()) {
    getIt.unregister<GlobalNotificationService>();
  }

  getIt.registerSingleton<GlobalNotificationService>(
    GlobalNotificationService(
      getIt<NotificationWebSocketDataSource>(),
      getIt<ProfileService>(),
    ),
    dispose: (service) => service.dispose(),
  );
}

void _registerDepositFeature() {
  // All deposit classes are now auto-registered by getIt.init()
  // No manual registration needed
}

void _registerSpotDepositFeature() {
  // All SPOT deposit classes are now auto-registered by getIt.init()
  // No manual registration needed
}

void _registerEcoDepositFeature() {
  // All ECO deposit classes are now auto-registered by getIt.init()
  // No manual registration needed
}

void _registerFuturesDepositFeature() {
  // All FUTURES deposit classes are now auto-registered by getIt.init()
  // No manual registration needed
}

void _registerFuturesFeature() {
  // Futures feature dependencies are now auto-registered by @injectable annotations
  // No manual registration needed
}

void _registerTransferFeature() {
  // Transfer feature dependencies are now auto-registered by @injectable annotations
  // No manual registration needed
}

void _registerChartFeature() {
  // Chart feature dependencies are now auto-registered by @injectable annotations
  // No manual registration needed
}

void _registerTradeFeature() {
  // TradingChartBloc is auto-registered with @injectable annotation
  // AiInvestmentBloc is also auto-registered with @injectable annotation
  // No manual registration needed
}

void _registerStakingFeature() {
  // StakingBloc and PositionBloc are now auto-registered with @injectable annotations
  // No manual registration needed
}

void _registerBlogFeature() {
  // Register BlogBloc if not already
  if (!getIt.isRegistered<BlogBloc>()) {
    getIt.registerFactory(() => BlogBloc(getIt()));
  }

  // Register AuthorsBloc for authors feature
  if (!getIt.isRegistered<AuthorsBloc>()) {
    getIt.registerFactory(() => AuthorsBloc(getIt()));
  }
}

void _registerEcommerceFeature() {
  // All e-commerce classes are now auto-registered by @injectable annotations
  // Manual registration for CartBloc - use lazySingleton to maintain state
  if (!getIt.isRegistered<CartBloc>()) {
    getIt.registerLazySingleton(() => CartBloc(
          getCartUseCase: getIt<GetCartUseCase>(),
          addToCartUseCase: getIt<AddToCartUseCase>(),
          updateCartItemQuantityUseCase: getIt<UpdateCartItemQuantityUseCase>(),
          removeFromCartUseCase: getIt<RemoveFromCartUseCase>(),
          clearCartUseCase: getIt<ClearCartUseCase>(),
        ));
  }

  // Manual registration for ShopBloc
  if (!getIt.isRegistered<ShopBloc>()) {
    getIt.registerFactory(() => ShopBloc(
          getIt<GetProductsUseCase>(),
          getIt<GetCategoriesUseCase>(),
        ));
  }
}

void _registerP2PFeature() {
  // P2P feature dependencies are now auto-registered by @injectable annotations
  // No manual registration needed
}

void _registerSupportFeature() {
  // Most support classes are auto-registered by @injectable annotations
  // Manual registration for LiveChatBloc to ensure proper dependency injection
  if (!getIt.isRegistered<LiveChatBloc>()) {
    getIt.registerFactory(() => LiveChatBloc(getIt(), getIt<AuthBloc>()));
  }

  // Register TicketDetailBloc for normal ticket conversations
  if (!getIt.isRegistered<TicketDetailBloc>()) {
    getIt.registerFactory(() => TicketDetailBloc(getIt()));
  }
}

void _registerIcoFeature() {
  // ICO feature now uses real API integration
  // All ICO classes are auto-registered by @injectable annotations
  // No manual registration needed for auto-registered classes
}

void _registerCreatorFeature() {
  // Ensure single registration
  if (!getIt.isRegistered<CreatorRemoteDataSource>()) {
    getIt.registerLazySingleton<CreatorRemoteDataSource>(
        () => CreatorRemoteDataSourceImpl(getIt()));
  }

  if (!getIt.isRegistered<CreatorRepository>()) {
    getIt.registerLazySingleton<CreatorRepository>(
        () => CreatorRepositoryImpl(getIt(), getIt()));
  }

  if (!getIt.isRegistered<LaunchTokenUseCase>()) {
    getIt.registerLazySingleton<LaunchTokenUseCase>(
        () => LaunchTokenUseCase(getIt()));
  }

  if (getIt.isRegistered<CreatorBloc>()) {
    getIt.unregister<CreatorBloc>();
  }

  getIt.registerFactory(() => CreatorBloc(getIt(), getIt()));

  if (!getIt.isRegistered<GetLaunchPlansUseCase>()) {
    getIt.registerLazySingleton<GetLaunchPlansUseCase>(
        () => GetLaunchPlansUseCase(getIt()));
  }

  if (!getIt.isRegistered<LaunchPlanCubit>()) {
    getIt.registerFactory(() => LaunchPlanCubit(getIt()));
  }

  if (!getIt.isRegistered<GetInvestorsUseCase>()) {
    getIt.registerLazySingleton<GetInvestorsUseCase>(
        () => GetInvestorsUseCase(getIt()));
  }

  if (!getIt.isRegistered<GetCreatorStatsUseCase>()) {
    getIt.registerLazySingleton<GetCreatorStatsUseCase>(
        () => GetCreatorStatsUseCase(getIt()));
  }

  if (!getIt.isRegistered<InvestorsCubit>()) {
    getIt.registerFactory(() => InvestorsCubit(getIt()));
  }

  if (!getIt.isRegistered<StatsCubit>()) {
    getIt.registerFactory(() => StatsCubit(getIt()));
  }

  if (!getIt.isRegistered<GetCreatorPerformanceUseCase>()) {
    getIt.registerLazySingleton(() => GetCreatorPerformanceUseCase(getIt()));
  }

  if (!getIt.isRegistered<PerformanceCubit>()) {
    getIt.registerFactory(() => PerformanceCubit(getIt()));
  }
}

void _registerKycFeature() {
  // Manual registration for KYC dependencies to ensure proper injection order
  if (!getIt.isRegistered<KycRemoteDataSource>()) {
    getIt.registerLazySingleton<KycRemoteDataSource>(
      () => KycRemoteDataSourceImpl(getIt<DioClient>()),
    );
  }

  if (!getIt.isRegistered<KycRepository>()) {
    getIt.registerLazySingleton<KycRepository>(
      () => KycRepositoryImpl(getIt<KycRemoteDataSource>()),
    );
  }

  // Use cases are auto-registered via @injectable
  // Bloc is auto-registered via @injectable
}

void _registerThemeFeature() {
  // Theme feature dependencies are auto-registered by @injectable annotations
  // No manual registration needed - ThemeBloc, Use Cases, Repository, and DataSource are all auto-injected
}

void _registerNewsFeature() {
  // News feature dependencies are auto-registered by @injectable annotations
  // No manual registration needed for:
  // - NewsRemoteDataSource
  // - NewsLocalDataSource
  // - NewsRepository
  // - News use cases
  // - NewsBloc
}

void _registerMlmFeature() {
  // MLM/Affiliate feature dependencies are now auto-registered by @injectable annotations
  // Manual override for MlmRewardsBloc to ensure proper dependency injection
  if (getIt.isRegistered<MlmRewardsBloc>()) {
    getIt.unregister<MlmRewardsBloc>();
  }

  getIt.registerFactory(
    () => MlmRewardsBloc(
      getIt<GetMlmRewardsUseCase>(),
      getIt<ClaimMlmRewardUseCase>(),
    ),
  );

  // Manual registration for MlmNetworkBloc
  if (getIt.isRegistered<MlmNetworkBloc>()) {
    getIt.unregister<MlmNetworkBloc>();
  }

  getIt.registerFactory(
    () => MlmNetworkBloc(
      getIt<GetMlmNetworkUseCase>(),
    ),
  );
}
